import { promises as fs } from "fs";
import path from "path";
import * as Log from "../../build/output/log";
import isError from "../../lib/is-error";
/**
 * Check if a given file path is a directory or not.
 * @param {string} filePath The path to a file to check.
 * @returns {Promise<boolean>} `true` if the path is a directory.
 */ async function isDirectory(filePath) {
    try {
        return (await fs.stat(filePath)).isDirectory();
    } catch (error) {
        if (isError(error) && (error.code === "ENOENT" || error.code === "ENOTDIR")) {
            return false;
        }
        throw error;
    }
}
/**
 * Create a file with eslint output data
 * @param {string} outputFile The name file that needs to be created
 * @param {string} outputData The data that needs to be inserted into the file
 */ export async function writeOutputFile(outputFile, outputData) {
    const filePath = path.resolve(process.cwd(), outputFile);
    if (await isDirectory(filePath)) {
        Log.error(`Cannot write to output file path, it is a directory: ${filePath}`);
    } else {
        try {
            await fs.mkdir(path.dirname(filePath), {
                recursive: true
            });
            await fs.writeFile(filePath, outputData);
            Log.info(`The output file has been created: ${filePath}`);
        } catch (err) {
            Log.error(`There was a problem writing the output file: ${filePath}`);
            console.error(err);
        }
    }
}

//# sourceMappingURL=writeOutputFile.js.map